/*
Copyright (c) 2008, Lukas Winter
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    * Neither the name of the <ORGANIZATION> nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef MATH_HEADER
#define MATH_HEADER

#if HAVE_CONFIG_H
#include <../config/config.h>
#endif

#include <cmath>
#include <limits>
#include <boost/logic/tribool.hpp>

namespace trayce
{
  /*here should be put all definitions of mathematical functions
  needed for scalar. Multiple definitions for different types
  can be put here by using function overloading*/
  
  //everything from <cmath> for float, double and long double
  //see type.hpp for definition of scalar
  using std::cos;
  using std::acos;
  using std::cosh;
  using std::sin;
  using std::asin;
  using std::sinh;
  using std::tan;
  using std::atan;
  using std::atan2;
  using std::tanh;
  using std::pow;
  using std::exp;
  using std::sqrt;
  using std::log;
  using std::log10;
  using std::ceil;
  using std::floor;
  using std::abs;
  //using std::fabs;
  
  using namespace boost::logic; //for tribool
  
  /*null_type represents the type of a neutral value for a
  numeric type that can be used for scalar, i.e. all scalar types
  must be convertible from null_type, the opposite does not have
  to be true. All in-builts and all classes that can be converted
  implicitly from int already satisfy this need.*/
  enum null_type
  {
    zero = 0, //neutral value for addition
    one = 1  //neutral value for multiplication
  };
  
  template <typename T>
  T sgn(const T& x)
  {
    if(x == zero) { return x; } //preserves +/-0 floating point representation
    return x < zero?-one:one;
  }
  
  template <typename T>
  T sgn_no_zero(const T& x)
  {
    return x < zero?-one:one;
  }
  
  template <typename T>
  T cbrt(const T& x) //cube root, preserving sign
  {
    return pow(abs(x), (static_cast<T>(one))/3) * sgn(x);
  }
  
  template <typename T>
  T max(const T& a, const T& b)
  {
    return a > b? a:b;
  }
  
  template <typename T>
  T min(const T& a, const T& b)
  {
    return a < b? a:b;
  }
  
  /*
  * These class represent mathematical functions and are used by several
  * components for numerical calculations like integration or root-finding
  */
  
  template <typename X, typename Y>
  class math_function
  {
    public:
    Y virtual operator()(X) const = 0;
    
    struct sample
    {
      X x;
      Y f_x;
      sample(const math_function& f, X ix) :x(ix), f_x(f(x))
      {}
    };
    virtual ~math_function() {}
  };
  
  template <typename X, typename Y>
  class constant_math_function :public math_function<X, Y>
  {
    Y constant;
    public:
    Y virtual operator()(X) const
    { return constant; }
  };
  
  //several constants used for physical and mathematical calculations
  //note: http://physics.nist.gov/cuu/Constants/index.html
  const double pi = M_PI;
  const double planck_constant = 6.62606896e-34; //Joule seconds, CODATA 2006
  const double light_speed = 299792458; //metres per second, definition of the metre
  const double eulers_number = 2.7182818284590452354; //wikipedia
  const double boltzmann_constant = 1.3806504e-23; //Joule per Kelvin, CODATA 2006
  
  namespace constant //we do not want to pollute the trayce namespace with one-letter constants
  {
    //const double pi = ::trayce::pi;
    const double h = ::trayce::planck_constant;
    const double c = ::trayce::light_speed;
    const double e = ::trayce::eulers_number;
    const double k = ::trayce::boltzmann_constant;
  }
}

#endif //MATH_HEADER
