/*
Copyright (c) 2008, Lukas Winter
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    * Neither the name of the <ORGANIZATION> nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef MATRIX_HEADER
#define MATRIX_HEADER

#if HAVE_CONFIG_H
#include <../config/config.h>
#endif

#include <boost/operators.hpp>
#include <boost/array.hpp>
#include <boost/static_assert.hpp>
#include <boost/utility/enable_if.hpp>
#include "math.hpp"

namespace trayce
{
  //helper enum
  enum enabled_method
  {
    all,
    quadratic,
    quadratic_2,
    quadratic_3,
    quadratic_4
  };


  template <typename T, int m, int n>
  class basic_matrix :
    boost::addable< basic_matrix<T, m, n>,
    boost::subtractable< basic_matrix<T, m, n>,
    boost::multipliable< basic_matrix<T, m, n>, T,
    boost::dividable< basic_matrix<T, m, n>, T
    > > > >
  {
    public:
    //types of matrices
    enum type
    {
      null,
      identity
    };
    private:
    static boost::array<T, m*n> init_fields(const T& x)
    {
      boost::array<T, m*n> ret;
      ret.assign(x);
      return ret;
    }
    static boost::array<T, m*n> init(type t)
    {
      boost::array<T, m*n> ret;
      switch(t)
      {
        case null:
        ret = init_fields(zero);
        break;
        case identity:
        set_identity(ret);
        break;
      }
      return ret;
    }
    static void set_identity(boost::array<T, m*n>& x)
    {
      BOOST_STATIC_ASSERT(n == m);
      x.assign(zero);
      for(int i = 0; i < n; ++i)
      {
        x[i*m + i] = one; //cycle through elements diagonally
      }
    }
    
    //helper methods   
    template <enabled_method e>
    typename boost::enable_if_c<e==quadratic_2, T>::type do_determinant()
    {
      return data[0] * data[3] - data[1] * data[2];
    }
    
    template <enabled_method e>
    typename boost::enable_if_c<e==quadratic_3, T>::type do_determinant();
    
    template <enabled_method e>
    typename boost::enable_if_c<e==quadratic_4 || e==quadratic, T>::type do_determinant();
    
    template <enabled_method e>
    typename boost::enable_if_c<
      e==quadratic_2 || e==quadratic_3 || e==quadratic_4 || e==quadratic,
      basic_matrix<T, m, n> >::type do_adjugate();
    
    template <enabled_method e>
    typename boost::enable_if_c<e==quadratic_2, basic_matrix<T, m, n> >::type do_invert();
    
    template <enabled_method e>
    typename boost::enable_if_c<
      e==quadratic_3 || e==quadratic_4 || e==quadratic, basic_matrix<T, m, n> >::type
      do_invert();
    
    //members
    boost::array<T, m*n> data;
    public:
    //constructors
    basic_matrix() //depends on the user to initialize the values!
    {}
    explicit basic_matrix(type t)
      :data(init(t))
    {}
    //operators
    basic_matrix<T, m, n>& operator+=(const basic_matrix<T, m, n>&);
    basic_matrix<T, m, n>& operator-=(const basic_matrix<T, m, n>&);
    basic_matrix<T, m, n>& operator*=(const T&);
    basic_matrix<T, m, n>& operator/=(T);
    
    /*note: We won't offer a =* operator, because multiplications of matrices are not
    commutative and require a copy anyway. I think it will save us a lot of confusion.*/
    template <int other_m, int other_n>
    basic_matrix<T, other_m, n> operator*(const basic_matrix<T, other_m, other_n>&);
    //other operations
    void transpose();
    
    T determinant()
    {
      return do_determinant<
      m==n?
      (m==2?quadratic_2:
      (m==3?quadratic_3:
      (m==4?quadratic_4:
      quadratic))):
      all
      >();
    }
    
    basic_matrix<T, m, n> get_inverse()
    {
      return do_invert<
      m==n?
      (m==2?quadratic_2:
      (m==3?quadratic_3:
      (m==4?quadratic_4:
      quadratic))):
      all
      >();
    }
    void invert()
    {
      *this = get_inverse();
    }
    
    basic_matrix<T, m, n> get_adjugate()
    {
      return do_adjugate<
      m==n?
      (m==2?quadratic_2:
      (m==3?quadratic_3:
      (m==4?quadratic_4:
      quadratic))):
      all
      >();
    }
    void adjugate()
    {
      *this = get_adjugate();
    }
    basic_matrix<T, m-1, n-1> submatrix(std::size_t i, std::size_t j);
    
    //element access (not checked)
    T& element(std::size_t i, std::size_t j)
    { return data[j*m + i]; }
    const T& element(std::size_t i, std::size_t j) const
    { return data[j*m + i]; }
  };
}

#include "matrix.impl"

#endif //MATRIX_HEADER
