/*
Copyright (c) 2008, Lukas Winter
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    * Neither the name of the <ORGANIZATION> nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef VECTOR_HEADER
#define VECTOR_HEADER

#if HAVE_CONFIG_H
#include <../config/config.h>
#endif

//easy operator definitions
#include <boost/operators.hpp>
#include <boost/array.hpp>
#include <boost/static_assert.hpp>
#include "math.hpp"
#include "matrix.hpp"


namespace trayce
{
  template <typename T, int n>
  class basic_vector :
    boost::addable< basic_vector<T, n>,
    boost::subtractable< basic_vector<T, n>,
    boost::multipliable< basic_vector<T, n>, T,
    boost::dividable< basic_vector<T, n>, T
    > > > >
  {
    private:
    static boost::array<T, n> init_data(const T& x)
    {
      boost::array<T, n> ret;
      ret.assign(x);
      return ret;
    }
    static boost::array<T, n> fill_xyz(const T& x, const T& y, const T& z)
    {
      BOOST_STATIC_ASSERT(n == 3);
      boost::array<T, n> ret;
      ret[0] = x;
      ret[1] = y;
      ret[2] = z;
      return ret;
    }
    static boost::array<T, n> fill_xy(const T& x, const T& y)
    {
      BOOST_STATIC_ASSERT(n == 2);
      boost::array<T, n> ret;
      ret[0] = x;
      ret[1] = y;
      return ret;
    }
    
    //members
    boost::array<T, n> data;
    public:
    //constructors
    basic_vector()
      :data(init_data(zero)) //see math.hpp -> null_type for zero
    {}
    basic_vector(const T& i)
      :data(init_data(i))
    {}
    basic_vector(const T& x, const T& y, const T& z)
      :data(fill_xyz(x, y, z))
    {}
    basic_vector(const T& x, const T& y)
      :data(fill_xy(x, y))
    {}
    
    //operators
    basic_vector<T, n>& operator+=(const basic_vector<T, n>&);
    basic_vector<T, n>& operator-=(const basic_vector<T, n>&);
    basic_vector<T, n>& operator*=(const T&);
    basic_vector<T, n>& operator/=(const T&);
    basic_vector<T, n>& operator-();
    
    //other operations
    T dot(const basic_vector<T, n>&) const;
    basic_vector<T, n> cross(const basic_vector<T, n>&) const; //only works with n = 3
    T square_length() const;
    T length() const;
    basic_vector<T, n>& normalize();
    basic_vector<T, n>& normalise()
    { return normalize(); }
    
    //transformation
    basic_vector<T, n> transform(const basic_matrix<T, n, n>&);
    basic_vector<T, n> transform(const basic_matrix<T, n+1, n+1>&);
    basic_vector<T, n> transform_direction(const basic_matrix<T, n+1, n+1>&);
    
    //element access (not checked)
    T& operator[](std::size_t i)
    { return data[i]; }
    const T& operator[](std::size_t i) const
    { return data[i]; }
    
    //checking for null vector
    bool is_null() const;
  };
  
}

//implementation
#include "vector.impl"

#endif //VECTOR HEADER
